#!/bin/bash
#
#
# Copyright (c) 2010-2012 Ericsson AB.
# All rights reserved.
# Author: Dhirender Rathi(dhirender.rathi@ericsson.com)
#
# NOTE - this halt.sh is only for REAL RP
# The SSR-sim version is in asg-ase/system-files/etc/siara/ssrsim/W006/halt.sh
# Keep the two in sync

print_mesg() {
	TIME=$(date "+%b %d %H:%M:%S")
	echo ${TIME} ${1}
	echo "${TIME} ${1}" >>/var/log/boot.log
}

update_leds() {
    # Set operational LED to fast blink
    python -c 'from ctypes import *; \
               fpga = CDLL("libnhk_tc_fpga_cms.so"); \
               fpga.nhk_write_fpga_operational_led(3);'
}

stop_com_procs() {
	print_mesg "Stopping COM processes..."
	pids=`ps -ef|grep /opt/com | grep -v grep|cut -c10-14`
	if [ -n "$pids" ] ; then
		# Signal 15 doesn't kill it
		kill -9 $pids
	fi
}

stop_tcmd() {
   print_mesg "Stopping TCMD..."
   tcmd=`ps -ef|grep /usr/lib/siara/bin/tcmd | egrep -v -e 'grep' |cut -c10-14`
   if [ -n "$tcmd" ] ; then
      kill -15 $tcmd
   fi
   sleep 1
}

stop_ipos_drv_procs() {
	print_mesg "Stopping IPOS drv processes..."
	pids=`ps -ef|grep /usr/lib/siara/bin/cms | egrep -v -e 'grep' |cut -c10-14`
	if [ -n "$pids" ] ; then
		kill -15 $pids
	fi
	# Wait for them to finish exit procedure
	sleep 1
}


stop_ipos_procs() {
	print_mesg "Stopping IPOS processes..."
	pids=`ps -ef|grep /usr/lib/siara/bin| egrep -v -e 'grep|loggd|core_mgmt|gcore' |cut -c10-14`
	if [ -n "$pids" ] ; then
		kill -15 $pids
	fi
	# Wait for them to finish exit procedure
	sleep 2
}

stop_diag_procs() {
	print_mesg "Stopping IPOS diagnostics processes..."
	pids=`ps -ef|grep /usr/lib/siara/diags/bin| egrep -v -e 'grep|loggd|rpsw_util' |cut -c10-14`
	if [ -n "$pids" ] ; then
		kill -TERM $pids
	fi
	# Wait for them to finish exit procedure
	sleep 2
}

stop_loggd() {
	print_mesg "Stopping IPOS Logger Daemon..."
	pid=`ps -ef|grep /usr/lib/siara/bin/loggd |grep -v grep|cut -c10-14`
	if [ -n "$pid" ] ; then
		kill -TERM $pid
	fi
        i=0
        while [ $i -lt 10 ]; do
		pid=$(ps -ef | grep /usr/lib/siara/bin/loggd | grep -v grep | cut -c10-14)
		if [ -z $pid ]; then
			break
		fi
		printf "."
		sleep 1
		i=$((i+1))
	done
	if [ $i -eq 10 ]; then
	        print_mesg "Force stopping IPOS Logger Daemon..."
		pid=$(ps -ef | grep /usr/lib/siara/bin/loggd | grep -v grep | cut -c10-14)
		[ -n $pid ] && kill -9 $pid >/dev/null 2>&1
		sleep 1
	fi
}

stop_others() {
	print_mesg "Stopping other processes..."
	kill -9 $(ps -ef|grep /usr/bin/ntpd |grep -v grep|cut -c10-14) >/dev/null 2>&1
	kill -9 $(ps -ef|grep /usr/sbin/dhcpd |grep -v grep|cut -c10-14) >/dev/null 2>&1
	kill -9 $(ps -ef|grep /opt/services |grep -v grep|cut -c10-14) >/dev/null 2>&1
	kill -9 $(ps -ef|grep /opt/disk/services |grep -v grep|cut -c10-14) >/dev/null 2>&1
	kill -9 $(ps -ef|grep sleep |grep -v grep|cut -c10-14) >/dev/null 2>&1

	TIME=$(date "+%b %d %H:%M:%S")
	echo "  Process running at time ${TIME}" >>/var/log/boot.log
	ps -ef >>/var/log/boot.log

        kill -9 $(ps -ef|egrep -v -e 'grep|halt.sh|init|\]'|cut -c10-14) >/dev/null 2>&1

}


rm_ipcs() {
	print_mesg "Deleting shared memory segments..."
	for I in `ipcs -m | grep "^0x" | cut -c1-10`; do
        	ipcrm -M $I
	done
	print_mesg "Deleting semaphores..."
	for I in `ipcs -s | grep "^0x" | cut -c1-10`; do
        	ipcrm -S $I
	done
	print_mesg "Deleting message queues..."
	for I in `ipcs -q | grep "^0x" | cut -d" " -f2`; do
        	ipcrm -q $I
	done
}

wait_for_coredump() {

        i=0
	print_mesg "Checking if a core dump is in progress..."
	pids=$(ps -ef | egrep -e 'rpsw_util' |egrep -v grep|cut -c10-14)
	if [[ -n $pids ]]; then
		kill -15 $pids
	fi
	while [ $i -lt 60 ]; do
		pids=$(ps -ef | egrep -e 'core_mgmt|/usr/bin/gcore|rpsw_util' |egrep -v grep|cut -c10-14)
		if [ "x" == "x$pids" ]; then
			break
		else
			i=$((i+1))
			printf "."
			sleep 1
		fi
	done
}

wait_for_firmwareupdate() {
        i=0
        print_mesg "Checking if a firmware update is in progress..."
		  # Wait for 4 minutes to see if firmware upgrade is in progress
        while [ $i -lt 240 ]; do
                pids=$(ps -ef | egrep -e 'flash_upgrade|fpga_upgrade|flashrom|version_upgrade|cyclone_upgrade' |egrep -v grep|cut -c10-14)
                if [ "x" == "x$pids" ]; then
                        break
                else
                        i=$((i+1))
                        printf "."
                        sleep 1
                fi
        done
}

unload_kernel_mods() {

	print_mesg "Unloading kernel modules..."
	# moved on the top after EV 192529. Unload only if it is loaded.
#        modinfo tipc >/dev/null 2>&1
#        if [ $? -eq 0 ]; then
#            modprobe -r tipc
#        fi
	#modprobe -r bonding
	#modprobe -r 8021q
	modprobe -r rbn_nf_ipacl
	modprobe -r rbn_nf_ips
	modprobe -r rbn_nf_tcp
	# modprobe -r af_ipc
	modprobe -r af_rbn_netlink
	modprobe -r af_isis
	modprobe -r rbn_ext_ip_so
	#modprobe -r xcrp
	#modprobe -r rbn_platform
	# modprobe -r iflc
	# Do not need to unload the module
	# modprobe -r rbn_ctx_kmod

}

umount_fs() {

	# Umount all file-systems except root partition
	print_mesg "Stopping all remaining processes."
	kill -9 $(ps -ef | egrep -v -e'egrep|halt.sh|init|\['|cut -c10-14) >/dev/null 2>&1
	sync

	print_mesg "Unmounting file-systems..."
	grep /usr/lib/siara/diags /proc/mounts >/dev/null 2>&1 && umount -n /usr/lib/siara/diags
	grep /dev/shm /proc/mounts >/dev/null 2>&1 && umount  -n /dev/shm
	grep /md /proc/mounts >/dev/null 2>&1 && umount -n /md
	grep /var/tftpboot/lc /proc/mounts >/dev/null 2>&1 && umount -n /var/tftpboot/lc
	grep /var/tftpboot/diag /proc/mounts >/dev/null 2>&1 && umount -n /var/tftpboot/diag
	grep /var/tftpboot/etc /proc/mounts >/dev/null 2>&1 && umount -n /var/tftpboot/etc
	grep /var/opt/services/epg /proc/mounts >/dev/null 2>&1 && umount -n /var/opt/services/epg >/dev/null 2>&1
	grep /ram /proc/mounts >/dev/null 2>&1 && umount -n /ram
	grep /media/flash /proc/mounts >/dev/null 2>&1 && umount -n /media/flash
	grep /flash /proc/mounts >/dev/null 2>&1 && umount -n /flash
	grep /var /proc/mounts >/dev/null 2>&1 && umount -n /var
	grep /dev/dynattr /proc/mounts >/dev/null 2>&1 && umount -n /dev/dynattr
	grep /p01/boot /proc/mounts >/dev/null 2>&1 && umount -n /p01/boot
	grep /p01 /proc/mounts >/dev/null 2>&1 && umount -n /p01
	grep /p02/boot /proc/mounts >/dev/null 2>&1 && umount -n /p02/boot
	grep /p02 /proc/mounts >/dev/null 2>&1 && umount -n /p02
	[ -f /proc/bus/usb/devices ] && umount -n /proc/bus/usb
	umount -n /sys/kernel/debug

	# Umount cgroup mount points
	umount -n /sys/fs/cgroup/mem
	umount -n /sys/fs/cgroup

	umount -n /sys
	grep /dev/mqueue /proc/mounts >/dev/null 2>&1 && umount -n /dev/mqueue
	grep /dev/pts /proc/mounts >/dev/null 2>&1 && umount -n /dev/pts

	# Sync
	echo 3 > /proc/sys/vm/drop_caches
	echo "Syncing...."
        mount -n -oremount,ro /
	sync
	sync
        [ -L /sys/block/sda ] && /usr/bin/sdparm --command=sync /dev/sda >/dev/null 2>&1
        [ -L /sys/block/sdb ] && /usr/bin/sdparm --command=sync /dev/sdb >/dev/null 2>&1
        sleep 2
	if [ -h /dev/cms/chassis_reload ] ; then
		echo "Chassis-recovery reload in 2 seconds....."
		/usr/lib/siara/bin/tc_cpld_reset_tool -w 0x0c 0x0
		/usr/lib/siara/bin/tc_cpld_reset_tool -w 0x0c 0x10
		sleep 2
		echo "Chassis-recovery reload failed.. RP only reset is successful."
	fi
	echo "Done."
}

# Update RTC, etc..
do_housekeeping() {

	# Save random seed
	touch /var/lib/random-seed
	chmod 600 /var/lib/random-seed
	print_mesg "Saving random seed..."
	dd if=/dev/urandom of=/var/lib/random-seed count=1 bs=512 2>/dev/null

	# Sync the system clock.
	if [ -x /sbin/hwclock ]; then
		print_mesg "Syncing hardware clock to system time..."
		#log_hwclock "...Pre-sync"
		/sbin/hwclock --utc --systohc
		#log_hwclock "...Post-sync"
	fi
}

log_hwclock() {
	# Cross-checking with bypassing /sbin/hwclock.
	local RTC="$(/bin/date --date=@$(/bin/cat /sys/class/rtc/rtc0/since_epoch))"
	print_mesg "${1}: ${RTC} [$(/usr/bin/tr '\n' '`' 2>/dev/null < /etc/adjtime)]"
}

# Unconfigure VLAN
ifdown_net() {
	IFCONFIG=/sbin/ifconfig
	$IFCONFIG ethSw0 down >/dev/null 2>&1
	$IFCONFIG ethSw0.1 down >/dev/null 2>&1
	# vconfig rem ethSw0.1 >/dev/null 2>&1
}

get_board_revision() {
        BOARD_REVISION=$(cat /sys/firmware/rbn/nvenv/chosen/board-revision)
}
get_fpga_revision() {
        FPGA_REVISION=$(cat /sys/firmware/rbn/nvenv/chosen/sys-fpga-revision)
}

########################
## Script starts here ##
########################

#Cache board revision, need for reset
get_board_revision
get_fpga_revision

# Update LED state for deactivation
update_leds

# Stop TCMd process
stop_tcmd

# Kill all processes that accessing PCIe
stop_ipos_drv_procs

# Stop boot Services for other cards
print_mesg "Stopping boot_svcs..."
[ -f /etc/boot_svcs ] && /etc/boot_svcs stop >/dev/null 2>&1

# Update RTC time-stamp
do_housekeeping

# Wait for firmware update to complete, if running
wait_for_firmwareupdate

# Stop COM processes
stop_com_procs

# Stop diag processes
stop_diag_procs

# Stop IPOS processes except loggd
stop_ipos_procs

# Stop other processes
stop_others

# Unconfigure network stack
print_mesg "Unconfiguring network stack..."
ifdown_net

# Stop IPOS Logg daemon
stop_loggd

# Check for coredump
wait_for_coredump

# Start disk sync
print_mesg "Starting disk sync..."
echo 8192 >/proc/sys/vm/dirty_bytes
echo 8192 >/proc/sys/vm/dirty_background_bytes
sync

# Remove IPC stuff
rm_ipcs

# Final sanity-check on RTC
#if [ -x /sbin/hwclock ]; then
#    log_hwclock "Pre-rmmod"
#fi

# Unload kernel modules
unload_kernel_mods

# Unmount file-systems
umount_fs

