#!/bin/bash
#
# Copyright (c) 2015-2017 Ericsson AB.
# All rights reserved.
#
# FILE : /etc/init.sh for NHK-RP


HOSTNAME=Ericsson
NFSROOT=0
ALT_ROOT_PART="NFS"
ALT_ROOT_DIR="/p02"

BOOT_PART_LABEL="boot1"
ALT_BOOT_PART_LABEL="boot2"
ROOT_PART_LABEL="p01"
ALT_ROOT_PART_LABEL="p02"

IP=/sbin/ip
AWK=/usr/bin/awk
CAT=/bin/cat
IFCONFIG=/sbin/ifconfig
DIRNAME=/usr/bin/dirname
BASENAME=/usr/bin/basename
VCONFIG=/usr/sbin/vconfig
DYNATTR=/usr/lib/siara/bin/dynattr
SET_BOOT_PARTITION=/bin/set_boot_partition
TIPC_CONFIG=/usr/local/sbin/tipc-config
ETHTOOL=/usr/sbin/ethtool
CHAS_EEPROM_DYNATTR=/usr/lib/siara/bin/chas_inv_get

CARD_PRODUCT_CODE="W050"
DBG_MAC_LAST_BYTE="00"
MAC_LAST_BYTE="00"
HWNUM=0
CHASSIS_MAC="02:00:00:FF:00:00"
VIPP_MTU=1500
MAILBOX_1=1
CARD_UP=0x3000
TCRP_POD=/usr/lib/siara/bin/tcrp_pod
CHASSIS_ID=1
STACKING_ENABLE=0
GLOBAL_MTU=9200
BOOT_MODE="production"

# Utility functions

publish_sim_sensors() {
    mkdir -p /dev/cms/RP
    local id
    for id in {0..7}; do
        ln -s "40000" /dev/cms/RP/thermal_sensor_0$id
    done
    ln -s "0x00" /dev/cms/RP/device_presence_00
}

symlink_for_i2c() {
    mkdir -p /dev/cms/RP
    ln -s "0x211" /dev/cms/RP/cpld_revision_00
    ln -s "0x00" /dev/cms/RP/device_presence_00
}

get_slot_num() {
    echo $((1 + $CHASSIS_ID))
}

get_slot_ip() {
    local slot=$(get_slot_num)
    if [ $slot -eq 3 ]; then
        echo "253"
    else
        echo "252"
    fi
}

get_chid() {
    local hex=0;
    hex=$(ifconfig eth0 | grep HWaddr | $AWK '{print $5}' | $AWK '{print $4}' FS=:)
    echo $((0x$hex))
}

is_nhksim() {
    [ -d /nhksim ]
}


# Find string
strstr() {
    #return TRUE if $1 has $2, which is when the strip does remove things
    [ "${1#*$2*}" != "$1" ]
}

cmdline_has() {
    strstr "$($CAT /proc/cmdline)" "$1"
}

# Log Message on the console and boot log file.
logMesg() {
        TIME=$(date "+%b %d %H:%M:%S")
        echo $TIME $1 >/dev/console
        echo $TIME $1 >>/var/log/boot.log
}

# Rotate and Copy Log files: not managed by syslog or logrotate.
rotateLogFiles() {
        f=$(basename $1)
        [[ -f /var/log/${f}.4 ]] && mv -f /var/log/${f}.4 /var/log/${f}.5
        [[ -f /var/log/${f}.3 ]] && mv -f /var/log/${f}.3 /var/log/${f}.4
        [[ -f /var/log/${f}.2 ]] && mv -f /var/log/${f}.2 /var/log/${f}.3
        [[ -f /var/log/${f}.1 ]] && mv -f /var/log/${f}.1 /var/log/${f}.2
        [[ -f /var/log/${f} ]] && mv -f /var/log/${f} /var/log/${f}.1
}

# Set disk partitions name
list_disk_partitions() {

    grep "/p01 " /proc/mounts >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        ALT_ROOT_PART_LABEL="p01"
        ALT_ROOT_DIR="/p01"
        ALT_BOOT_PART_LABEL="boot1"
        ROOT_PART_LABEL="p02"
        BOOT_PART_LABEL="boot2"
    else
        ALT_ROOT_PART_LABEL="p02"
        ALT_ROOT_DIR="/p02"
        ALT_BOOT_PART_LABEL="boot2"
        ROOT_PART_LABEL="p01"
        BOOT_PART_LABEL="boot1"
    fi
}

# Get card product code
get_card_product_code() {

    if [ -f /sys/firmware/rbn/nvenv/chosen/board-product-code ]; then
        product_code=$(cat /sys/firmware/rbn/nvenv/chosen/board-product-code)
        [[ x"$product_code" != "x" ]] && CARD_PRODUCT_CODE=$product_code
    else
        logMesg "Warning: File /sys/firmware/rbn/nvenv/chosen/board-product-code not found."
    fi

    logMesg "Card product code is ${CARD_PRODUCT_CODE}"
}

#copy the rate-limit config file
card_ratelimit_copy() {
    if [ ! -f /flash/card_ratelimit.cfg ]; then
        /bin/cp /usr/lib/siara/bin/card_ratelimit.cfg /flash/card_ratelimit.cfg;
        chmod 660 /flash/card_ratelimit.cfg
    fi
}

# Copy out NVRAM Console logs
nvlog_hardware_console() {
    _hc_name=hwconsole-freeze
    _hc_log=/var/log/hwconsole-freeze
    _hc_done=b71dc104
    _hc_ctr=0
    _hc_dev=
    _hc_sysfs=
    _d=

    rotateLogFiles ${_hc_name}

    # get device name
    _d=`grep ${_hc_name} /proc/nvram | cut -f1`
    if [ $? -ne 0 ]; then
        logMesg "No HW Console Buffer: ${_hc_name}"
        return 1
    fi

    # Create device name
    _hc_sysfs=/sys/devices/platform/se-nvram/asg_nvram/${_d}
    _hc_dev=/dev/${_d}
    if [ ! -d ${_hc_sysfs} ]; then
        logMesg "No HW Console Buffer: ${_hc_sysfs}"
        return 1
    fi
    _maj=`cat ${_hc_sysfs}/dev | cut -d: -f1`
    _min=`cat ${_hc_sysfs}/dev | cut -d: -f2`
    mknod ${_hc_dev} c ${_maj} ${_min}

    # Make sure hardware console freeze buffer exist
    if [ ! -c ${_hc_dev} ]; then
        logMesg "Error: No hardware console buffer: ${_hc_dev}"
        return 1
    fi

    # Read the control flag, confirm the location with HW
    _hc_ctr=`od --skip-bytes=15356 --read-byte=4 -t x4 ${_hc_dev} | awk '{ print $2}'`
    if [ "${_hc_ctr}" != "${_hc_done}" ]; then
        return 1
    fi

    logMesg "Dump HW Console Buffer to ${_hc_log}";

    # Dump hardware console freeze buffer in file
    dd if=${_hc_dev} of=${_hc_log} bs=1 >/dev/null 2>&1

    # Reset the control flag
    dd if=/dev/zero of=${_hc_dev} bs=1 seek=15356 count=4 >/dev/null 2>&1
}

# Mount/Remount basic psuedo file-systems
mount_fs_linux() {

    rm -f /etc/mtab

    mount -t proc /proc /proc || logMesg "Error: Failed to mount /proc"

    # Find list of /boot, /p01, etc
    list_disk_partitions

    mount -t sysfs /sys /sys || logMesg "Error: Failed to mount /sys"

    mount -t debugfs none /sys/kernel/debug || logMesg "Error: Failed to mount /sys/kernel/debug"

    mount -o mode=0755 -t devtmpfs none /dev || logMesg "Error: Failed to mount /dev"

    mkdir -m 0755 /dev/shm || logMesg "Error: Failed to create /dev/shm"

    mkdir -m 0755 /dev/pts || logMesg "Error: Failed to create /dev/pts"

    mount -t devpts none /dev/pts >/dev/null 2>&1 || logMesg "Error: Failed to mount devpts"

    mkdir -m 0755 /dev/mqueue || logMesg "Error: Failed to create directory /dev/mqueue"

    mount -t mqueue none /dev/mqueue || logMesg "Error: Failed to mount /dev/mqueue"

    #FSH: remove the sgid for /tmp
    mount -o mode=1777 -t tmpfs none /tmp || logMesg "Error: Failed to mount /tmp"


    # Remount partitions to put entries into /etc/mtab as some code still looks at it
    mount -o remount,noatime,acl -L ${ROOT_PART_LABEL} /

    # Remount alternate root also read-only in production releases
    mount -o remount -L ${ALT_ROOT_PART_LABEL} ${ALT_ROOT_DIR}

    # Re-mount /boot(s)
    mount -o remount,sync -L ${BOOT_PART_LABEL} /boot
    mount -o remount,sync -L ${ALT_BOOT_PART_LABEL} ${ALT_ROOT_DIR}/boot

    mount -o remount,acl,commit=0 -L var /var
    chmod +t /var/log/

    # Add group ownership and set permissions of /flash
    # then mount with grpid option.
    chgrp 666 /flash
    chmod 3775 /flash
    mount -o remount,acl,grpid,commit=0 -L flash /flash
    chown root:666 /flash
    setfacl -m u:nobody:rx /flash
    chgrp 666 /flash/.ssh
    chmod 775 /flash/.ssh

}


#################################
# Set up file-systems for Linux #
#################################
setup_fs_linux() {

    # Hack to fix file ownership
    ls -ld /etc | grep root >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        /bin/chown -R root:root / >/dev/null 2>&1
    fi

    # Another Hack to Make sure required /var dirs exist
    for d in log run tftpboot ; do
        [ -d /var/$d ] || mkdir -p /var/$d
    done

    #setup tftpboot directory.
    (cd /var/tftpboot
        rm -f lc etc diag 2>/dev/null
        mkdir -p lc
        mkdir -p etc
        mkdir -p diag
        mount --bind /lc lc
        mount --bind /diag diag
        mount --bind /etc etc
    )

    # Mount pseudo partitions
    mount_fs_linux

    # Clean up /var.
    rm -rf /var/lock/cvs/* /var/run/screen/*
    find /var/lock /var/run ! -type d -exec rm -f {} \;
    [[ ! -d /var/lock ]] && mkdir -p /var/lock
    chmod 755 /var/lock
    [[ ! -d /var/run ]] && mkdir -p /var/run
    chmod 755 /var/run
    rm -f /var/lib/rpm/__db* &> /dev/null

    # Clean up /var/utmp and /var/wtmp
    > /var/run/utmp
    touch /var/log/wtmp
    chgrp utmp /var/run/utmp /var/log/wtmp
    chmod 0664 /var/run/utmp /var/log/wtmp

    > /var/run/utmpx
    touch /var/log/wtmpx
    chgrp utmp /var/run/utmpx /var/log/wtmpx
    chmod 0664 /var/run/utmpx /var/log/wtmpx

    # (which sets it to ugo:r).
    touch /tmp/shmemLibSegFile
    chmod 444 /tmp/shmemLibSegFile

    cmdline=$($CAT /proc/cmdline)
    if strstr "$cmdline" root=/dev/nfs ; then
        NFSROOT=1
        mount -o remount /
    fi

    # Create required /flash/home directory,
    # then change group & permissions
    [ ! -d /flash/home ] && mkdir /flash/home
    chgrp 666 /flash/home
    chmod 775 /flash/home
    chmod 777 /flash

    # Setup environment for udevd
    ln -snf /proc/self/fd /dev/fd
    ln -snf /proc/self/fd/0 /dev/stdin
    ln -snf /proc/self/fd/1 /dev/stdout
    ln -snf /proc/self/fd/2 /dev/stderr
    ln -snf /proc/kcore /dev/core
    ln -s /sbin/MAKEDEV /dev/MAKEDEV
    mkdir -p /dev/.udev/makedev.d
    /sbin/MAKEDEV -x ttyS0 tty1 tty2 tty3 tty4 tty5 tty6 ptyp0 ptmx\
         loop0 loop1 loop2 loop3 loop4 loop5 loop6 loop7 \
         ppp console null zero kmsg mem sda sdb sdb1 sdb5 sdb7 \
         sdb8 urandom random

    # Need /dev/mem and /dev/kmsg early
    [ -f /dev/kmsg ] || /sbin/MAKEDEV -x kmsg
    [ -f /dev/mem ] || /sbin/MAKEDEV -x mem

    rm -fr /dev/.udev

    # Disable console logging except for critical messages
    echo 1 >/proc/sys/kernel/printk

    # Disable sysrq
    echo 0 >/proc/sys/kernel/sysrq

    # Dump dmesg data in file
    rotateLogFiles dmesg
    dmesg -s 131072 > /var/log/dmesg

    logMesg "Updating shared lib paths..."
    /sbin/ldconfig  >/dev/null 2>&1
    /sbin/depmod -a

    # Mount diag fs at the end
    logMesg "Mounting diagnostics file-system..."
    DIAGS_IMG="/diag/diagfs"
    [ ! -d /usr/lib/siara/diags ] && mkdir -p /usr/lib/siara/diags
    if [ -f ${DIAGS_IMG} ]; then
        local retstr=$(/sbin/fsck.ext2 -p ${DIAGS_IMG} 2>&1)
        # return error only on unrecoverable filesystem errors
        [ $? -ge 2 ] && logMesg "Error: fsck failed for diagnostics file-system. ${retstr}" && return 1
        mount $DIAGS_IMG /usr/lib/siara/diags -t ext2 -o loop
        [ $? -ne 0 ] && logMesg "Error: Failed to mount diag file-system." && return 1
    fi

    DIAGS_EXT_IMG="/diag/nhk-rp-diag-ext-fs"
    [ ! -d /usr/lib/siara/diags/extra ] && mkdir -p /usr/lib/siara/diags/extra
    if [ -f ${DIAGS_EXT_IMG} ]; then
        local retstr=$(/sbin/fsck.ext2 -p ${DIAGS_EXT_IMG} 2>&1)
        # return error only on unrecoverable filesystem errors
        [ $? -ge 2 ] && logMesg "Error: fsck failed for extra diags file-system. ${retstr}" && return 1
        mount $DIAGS_EXT_IMG /usr/lib/siara/diags/extra -t ext2 -o loop
        [ $? -ne 0 ] && logMesg "Error: Failed to mount diag extra file-system." && return 1
    fi

    # Create control group infra for use with IPOS processes
    if ! grep cgroup_root /proc/mounts; then
        mount -t tmpfs cgroup_root /sys/fs/cgroup
        [ $? -ne 0 ] && logMesg "error: mount cgroup_root" && return 1
    fi

    # Create control group memory subsystem for IPOS PI and PD processes
    if [ ! -d /sys/fs/cgroup/mem ]; then
        mkdir /sys/fs/cgroup/mem
        [ $? -ne 0 ] && logMesg "error: mkdir /sys/fs/cgroup/mem"

        mount -t cgroup -o memory none /sys/fs/cgroup/mem
        [ $? -ne 0 ] && logMesg "error: mount memory cgroup" && rmdir /sys/fs/cgroup/mem && return 1
        mkdir /sys/fs/cgroup/mem/ipos_grp
        [ $? -ne 0 ] && logMesg "error: mkdir /sys/fs/cgroup/mem/ipos_grp" && return 1
    fi

}

#############################################
# Setup file-system related stuff for IPOS ##
#############################################
setup_fs_ipos() {


    # Create needed directories for RBOS and platform apps and File System Hardening feature
    [ -d /tmp/run ] || mkdir /tmp/run
    [ -d /tmp/lock ] || mkdir /tmp/lock
    [ -d /tmp/status ] || mkdir /tmp/status
    [ -d /var/md ] || mkdir /var/md
    [ -d /var/pod ] || mkdir /var/pod
    [ -d /var/service-pools ] || mkdir /var/service-pools
    [ -d /var/vsmd ] || mkdir /var/vsmd

    if [ ${NFSROOT} -eq 0 ]; then
        # Change group & permissions /md
        chmod 3770 /var/md/.
        chown root:666 /var/md/.
        mount --bind /var/md /md
        setfacl -m u:nobody:rx /md
    fi

    # Only one should be sufficient here
    [ ! -f "/release.inf" ] && touch /release.inf
    [ ! -f "/usr/lib/siara/bin/release.inf" ] && touch /usr/lib/siara/bin/release.inf
    [ ! -f /md/release.inf ] && touch /md/release.inf
    # Add group ownership and set permissions of /flash
    # then mount with grpid option.
    chgrp 666 /flash
    chmod 3775 /flash
    mount -o remount,grpid,commit=0 ${FLASH_DISK_PART} /flash
    chown root:666 /flash
    setfacl -m u:nobody:rx /flash
    # ssh2_login_banner is created by RCM
    if [ -f /etc/issue ]; then
        rm -f /etc/issue
        (cd /etc; ln -s ssh2_login_banner issue >/dev/null 2>&1)
    fi

    # change group ownership of /etc/siara directories
    [ -d /p01/etc/siara ] && chgrp 666 /p01/etc/siara && chmod go-w /p01/etc/siara
    [ -d /p02/etc/siara ] && chgrp 666 /p02/etc/siara && chmod go-w /p02/etc/siara

    if [ ! -L /etc/siara/logrtat.conf ]; then
        (cd /etc/siara; ln -s /etc/logrtat.conf logrtat.conf)
    fi

    # Create required /flash/.ssh directory,
    # then change ownership, group, & permissions
    [ ! -d /flash/.ssh ] && mkdir /flash/.ssh
    chgrp 666 /flash/.ssh
    chmod 775 /flash/.ssh

    # Create required /flash/home directory,
    [ ! -d /flash/home ] && mkdir /flash/home
    # Mount tmpfs to be used for CDR
    CDR_FS="/ram"

    [ ! -d $CDR_FS ] && mkdir -m 0755 $CDR_FS
    mount -t tmpfs -o mode=755,size=16m none $CDR_FS

    # Add directories used to store card specific information
    [[ ! -d /var/tftpboot/run/card ]] && mkdir -p /var/tftpboot/run/card

    # Delete existing files so that they get are freshly created with configuration
    [[ -d /var/tftpboot/run/card ]] && rm -f /var/tftpboot/run/card/*

    # change exec_cli mode; allow users to execute it as root
    chmod u+s /usr/lib/siara/bin/exec_cli
    chmod u+s /usr/lib/siara/bin/exec_cli_legacy
    chmod u+s /usr/lib/siara/bin/ipos_cli
    chmod u+s /usr/lib/siara/bin/uni_cli
    chmod u+s /usr/lib/siara/bin/flash_update
    chmod u+s /etc/format_udrv
    chmod 4755 /bin/su
    chmod u+s /usr/lib/siara/bin/se_ping
    chmod 4755 /bin/su
    # HACK: until rootfs is fixed: sigh..
    chmod u+s /usr/bin/ping
    chmod u+s /usr/bin/ping6

    # change some file permissions for security concern
    chmod 400 /etc/ssh/sshd_crypto_pool || true
    chmod 400 /etc/ssh/sshd_crypto_default || true
    chmod 400 /etc/ssh/sshd_config_shadow || true
    chmod 600 /etc/ssh/sshd_config || true
    chmod 640 /etc/shadow || true

    #FSH remove the suid/sgid for /sbin files.
    chmod u-s,g-s /sbin/{mke2fs,mkfs.ext2,mkfs.ext4dev,mkfs.ext3,mkfs.ext4}

    #Set permission for sudoers file
    chmod 440 /etc/sudoers

    #Set suid for /usr/bin/sudo
    chmod 4111 /usr/bin/sudo


    # Ensure that services and service-pools directories exist
    [[ ! -d /opt/disk/services ]] && mkdir -p /opt/disk/services
    [[ ! -d /opt/disk/service-pools ]] && mkdir -p /opt/disk/service-pools

    # Remove leftover DLM signature file from previous boot
    # from the alternate partition.
    [ -d /p02 ] && [ -f /p02/.inuse ] && rm -f /p02/.inuse
    [ -d /p01 ] && [ -f /p01/.inuse ] && rm -f /p01/.inuse
    # Ensure .inuse file exist on current parition we boot from
    [ ! -f /.inuse ] && echo >/.inuse

    card_ratelimit_copy
}


###################################
## Initialize Dynamic Attributes ##
###################################
set_dynattr() {

    [ -z "$1" ] && logMesg "set_dynattr: no arg1" && echo "" && return 1
    [ -z "$2" ] && logMesg "set_dynattr: no arg2" && echo "" && return 1

    local v=$(${DYNATTR} --set $1 $2)
    if [ $? -ne 0 ]; then
    logMesg "set_dynattr --set $1 $2 returned non-zero value. $v"
    fi
    return 0
}

get_dynattr() {

    [ -z "$1" ] && logMesg "get_dynattr: no arg" && echo "" && return 1
    local v=$(${DYNATTR} --get $1)
    [ -z "$v" ]  && logMesg "get_dynattr($1): no val" && echo "" && return 1

    echo $v
    return 0
}

#publish chassis inventory eeprom
publish_eeprom() {
    if [ -x ${CHAS_EEPROM_DYNATTR} ]; then
        ${CHAS_EEPROM_DYNATTR}
    fi
}

init_dynattr() {

    [ ! -e ${DYNATTR} ] && logMesg "${DYNATTR}: does not exist" && return 1
    [ ! -x ${DYNATTR} ] && logMesg "${DYNATTR}: not executable" && return 1

    # Initialize dynamic attr and symlink this card-type's personality
    rm -rf /dev/dynattr
    ${DYNATTR} --init-only
    [ "$?" -ne 0 ] && logMesg "${DYNATTR}: exit error $?" && return 1

    # Populate essiential values
    logMesg "Updating Dynamic Attributes..."

    local ma="d0:f0:f0:00:00:00"
    # TODO: Remove these fake settings when P1C is officially decommissioned
    set_dynattr chassis/mac_address $ma
    set_dynattr chassis/product_code W050
    set_dynattr chassis/serial_number X999988888
    # The following line shall be moved, once POD result is/are available.
    set_dynattr rpsw/is_master_capable 3

    # The slot_id mapping for standalone/stacked
    set_dynattr chassis/slot $(get_slot_num)

    if is_nhksim ; then
       set_dynattr ssrsim/true 1
       ma=$(cat /sys/class/net/eth0/address)
       set_dynattr chassis/mac_address $ma
       set_dynattr chassis/product_code W050
       set_dynattr chassis/serial_number X999988888
       set_dynattr card/product_code V001
       set_dynattr chassis/backplane_id 1
       set_dynattr chassis/fabric_speed 0
       set_dynattr chassis/version 10 # card-type
    else
       set_dynattr chassis/mac_address $(cat /sys/firmware/rbn/nvenv/mac-address)
       set_dynattr chassis/product_code $(cat /sys/firmware/rbn/nvenv/chosen/board-product-code)
       set_dynattr chassis/serial_number $(cat /sys/firmware/rbn/nvenv/eeprom/Board\ Serial\ Number)
       set_dynattr card/product_code V001
       set_dynattr chassis/backplane_id 1
       set_dynattr chassis/fabric_speed 0
       set_dynattr chassis/version 10 # card-type
    fi
    publish_eeprom
}

########################################
# Configure Core-Dump Capture Behavior #
########################################
config_core_mgmt() {

    if [ $NFSROOT -eq 1 ]; then
        [ -f /var/md/.core_count ] && echo 0 >/var/md/.core_count
        # Limit number of core file to not overwhelm boot server in the labs
        echo "| /usr/lib/siara/bin/core_mgmt %e.%p.%t.%h.core.gz net" >/proc/sys/kernel/core_pattern
    else
        # For now do not use pipe method to manage core files, until the bug fix.
        # Name Format is executable-name:PID:Time:Hostname.core
        echo "| /usr/lib/siara/bin/core_mgmt %e.%p.%t.%h.core.gz disk" >/proc/sys/kernel/core_pattern
    fi

    # Limit Number of parallel core-dumps
    echo "8" >/proc/sys/kernel/core_pipe_limit

    # Compress kernel core files, which are save uncompressed by crash-dump capture kernel to save time.
    kernel_cores=$(ls -1 /var/md/vmcore.* 2>/dev/null | grep -v gz)
    [ ! -z $kernel_cores ] && /bin/gzip $kernel_cores >/dev/null 2>&1 &
}

######################
## Configure Linux  ##
######################
configure_linux() {

    # Configure core dump management
    config_core_mgmt

    # Initialize hardware
    sysctl -w kernel.modprobe="/sbin/modprobe" >/dev/null 2>&1

    # Configure kernel parameters
    sysctl -e -p /etc/sysctl.conf >/dev/null 2>&1

    #Ephemeral Port Range is 33768 to 61000
    #Port range 32768 to 33768 for service application(define in rbos_services.h)
    sysctl -w net.ipv4.ip_local_port_range="33768 61000" >/dev/null 2>&1

    # Initialize pseudo-random number generator
    if [ -f "/var/lib/random-seed" ]; then
        $CAT /var/lib/random-seed > /dev/urandom
    else
        #[ "$READONLY" != "yes" ] && touch /var/lib/random-seed
        touch /var/lib/random-seed
    fi

    chmod 600 /var/lib/random-seed
    dd if=/dev/urandom of=/var/lib/random-seed count=1 bs=512 2>/dev/null
    [ "$?" -ne 0 ] && logMesg "Error: Failed to copy urandom seed."

    # Use the hardware RNG to seed the entropy pool, if available
    [ -x /sbin/rngd -a -c /dev/hw_random ] && rngd

    # Generate DSA/RSA keys for sshd
    DSA_KEY=/flash/.ssh/ssh_host_dsa_key;
    KEYGEN=/usr/bin/ssh-keygen;
    if [ ! -s $DSA_KEY ]; then
        logMesg $"Generating SSH2 DSA host key: ";
        rm -rf $DSA_KEY*
        if $KEYGEN -q -t dsa -f $DSA_KEY -C '' -N '' >&/dev/null; then
            chmod 400 $DSA_KEY;
            chmod 644 $DSA_KEY.pub;
        else
            logMesg "Error: Failed to generate ssh DSA keys..."
            return 1
        fi;
    fi

    RSA_KEY=/flash/.ssh/ssh_host_rsa_key;
    if [ ! -s $RSA_KEY ]; then
        logMesg $"Generating SSH2 RSA host key: ";
        rm -rf $RSA_KEY*
        if $KEYGEN -q -t rsa -f $RSA_KEY -C '' -N '' >&/dev/null; then
            chmod 400 $RSA_KEY;
            chmod 644 $RSA_KEY.pub;
        else
            logMesg "Error: Failed to generate ssh RSA keys..."
        fi;
    fi

    # test if sshd is able to run

    if ! sshd -t; then
        # try to run with the default sshd_config file
        mv -f /etc/ssh/sshd_config /etc/ssh/sshd_config_wrong
        cp /etc/ssh/sshd_config_shadow /etc/ssh/sshd_config
        chmod 600 /etc/ssh/sshd_config

        if ! sshd -t; then
            logMesg "sshd can not operate even with the default sshd_config file"
            # restore original file
            mv -f /etc/ssh/sshd_config_wrong /etc/ssh/sshd_config
        else
            logMesg "sshd could only operate with the default sshd_config file"
        fi
    else
        logMesg "sshd tested, OK"
    fi

    #Remove stacking symlink if configured. This branch do not have stacking enabled.
	# Not needed in stacking branch
    # if [ -L /var/local/chassis_id ]; then
    #    echo "Removing stacking configuration... /var/local/chassis_id"
    #    rm -rf /var/local/chassis_id
    #fi

}

###############################
# Setup IPOS/RBOS environment #
###############################
rbos_init() {

    # Create conf file for target APIs for NHK-RP
    if [ ! -f /etc/target_api.conf ]; then
    $CAT > /etc/target_api.conf <<+++
target_get_ofw_version  Nighthawk Firmware Version 1.0
target_get_mkernel_version Nighthawk Mini-Kernel Version 1.01
target_get_bootargs console=ttyS0
target_get_bootdev  bootdev=eth0
target_get_bootpath root=/dev/sda
target_get_reboot_reason 4
+++
    fi


    # Load RBOS kernel modules
    modprobe rbn_platform
    modprobe xcrp
    modprobe af_ipc
    modprobe af_rbn_netlink
    modprobe af_isis
    # modprobe rbn_ext_ip_so
    modprobe rbn_ctx_kmod
    modprobe rbn_nf_ips
    modprobe rbn_nf_tcp
    modprobe rbn_nf_ipacl
    modprobe sctp
    modprobe rbn_nf_tcp_nsr

    # In final version, do not run it in diag mode
    modprobe tipc

    modprobe ipv6

    #Load i2c kernel module
    modprobe i2c-dev && modprobe i2c-i801

    # enable PATH MTU in RP
    echo 0 > /proc/sys/net/ipv4/ip_no_pmtu_disc


    # Load FUSE kernel modules on RP
    modprobe fuse
    modprobe xt_recent
    modprobe xt_state

    # Setup Syspage values
    echo 500000000 >  /proc/sys/kernel/shmmax
    echo 600000 >  /proc/sys/net/sctp/valid_cookie_life

    # RBOS defined
    echo 500000000 >  /proc/sys/kernel/shmmax
    echo 2000 > /proc/sys/net/unix/max_dgram_qlen
    echo 1024 > /proc/sys/net/ipv4/neigh/default/gc_thresh1
    echo 4096 > /proc/sys/net/ipv4/neigh/default/gc_thresh2
    echo 8192 > /proc/sys/net/ipv4/neigh/default/gc_thresh3
    echo 0 > /proc/sys/net/ipv4/tcp_syncookies
    echo 1 > /proc/sys/net/ipv4/tcp_mtu_probing
    echo 1 >/proc/sys/net/ipv6/bindv6only

    # make sure we support sufficient number of IPv6 routes
    # each context consumes some IPv6 routes (4 right now)
    echo 100000 > /proc/sys/net/ipv6/route/max_size

    if is_nhksim ; then
        echo "Copying pm.conf & dl_manifest files ..."
        rm -f /etc/siara/pm.conf
        cp /etc/siara/pm.conf.nhksim /etc/siara/pm.conf
        rm -f /etc/siara/flash_upgrade.inf
        cp /etc/siara/dlm_manifest.nhksim /etc/siara/flash_upgrade.inf
    fi
    # update HW watchdog timer's before POD, just in case if pod
    # hangs processor, we have correct HW watchdog settings.
    # Select RP CPLD device
    /usr/sbin/i2cset -y 0 0x71 0x08

    # Set NMI watchdog to 3 seconds i.e. 6 (CPLD counter is for 500ms)
    /usr/sbin/i2cset -y 0 0x20 0x0a 0x06
    /usr/sbin/i2cset -y 0 0x20 0x0b 0x00
    # Set Hard Reset to 600 seconds i.e. 1200 (0x4B0)
    /usr/sbin/i2cset -y 0 0x20 0x08 0xB0
    /usr/sbin/i2cset -y 0 0x20 0x09 0x04

    # Now close the switch for CPLD.
    /usr/sbin/i2cset -y 0 0x71 0x00
}


rbos_init_syspage() {

    logMesg "Publishing IPOS syspage values"

    local a
    for a in version backplane_id product_code; do
        echo $(get_dynattr chassis/$a) > /sys/devices/platform/rbn/chassis/$a
    done
}

#update pod result to nvram
update_pod_status() {
    local status=$1;
    local cms_nv_dev=$(ls /dev/$(awk '$NF=="cms-block"{print $1;}' /proc/nvram))
    local rp_value=$(dd if=/dev/mem skip=$((0xfa17C300)) iflag=skip_bytes \
                    count=6 bs=2 | hexdump -C \
                    |  head -n1  | awk '{print $10 $11}')
    local rp_value=$((0x$rp_value & 0xfffc))
    local value=$(($rp_value + 4))
    local pod_status=$(($status & 0x3))
    local value=$(($value | $pod_status))
    local pod_result=$(printf "%04x" $value)
    printf '\x'${pod_result:0:2}'\x'${pod_result:2:2} \
           | dd of=${cms_nv_dev} bs=2 seek=5 count=1 conv=notrunc
}

######################################
# Perform platform specific sequence #
######################################
platform_init() {

    logMesg "Starting platform specific initialization"

    # Remove mmap files, if any
    rm -f /md/*.mmap

    # modprobe msr
    local SERIAL_NO="N/A"

    BOOT_MODE=$(cat /sys/firmware/rbn/nvenv/boot-mode)
    [ -f /sys/firmware/rbn/nvenv/chosen/boot-mode ] && \
                BOOT_MODE=$(cat /sys/firmware/rbn/nvenv/chosen/boot-mode)

    # Read RP Mailbox 6 register and check if we have to put RP in OSD mode MSB=1
    local rp_mode=$(dd if=/dev/mem skip=$((0xA000E)) iflag=skip_bytes  \
                    count=1 bs=2 | hexdump -C \
                    |  head -n1  | awk '{print $3 $2}')
    # 32768 = 0x8000 (check MSB in RP FPGA mailbox 6)
    if [ $((0x$rp_mode & 0x8000)) -eq 32768 ]; then
        BOOT_MODE="diag"
    fi

    [ -f /sys/firmware/rbn/nvenv/pod-level ] && \
                POD_LEVEL=$(cat /sys/firmware/rbn/nvenv/pod-level)

    [ -f /sys/firmware/rbn/nvenv/chosen/pod-level ] && \
                POD_LEVEL=$(cat /sys/firmware/rbn/nvenv/chosen/pod-level)

    [ -f "/sys/firmware/rbn/nvenv/eeprom/Board Serial Number" ] && \
                SERIAL_NO=$(cat "/sys/firmware/rbn/nvenv/eeprom/Board Serial Number")

    [ $BOOT_MODE = "diag" ] && POD_LEVEL=0

    # Initialize platform attributes..
    if init_dynattr; then
        rbos_init_syspage
    fi

    # Delete diags chassis attribute file if it exists.
    DIAG_CHASSIS_ATTR=/var/tftpboot/run/card/chassis_attr
    [ -f ${DIAG_CHASSIS_ATTR} ] &&  /bin/rm -f  ${DIAG_CHASSIS_ATTR}

    # Finally Invoke aegis_pod with only rpsw-pod callback
    # and the uio LKMs are loaded only if POD gives go-ahead.
    if [ -x ${TCRP_POD} -a ${POD_LEVEL} -ne 0 ]; then
        logMesg "Begin POD"
        ChassisID="$SERIAL_NO" ${TCRP_POD} --self --pod-level ${POD_LEVEL}
        ret=$?
        #now update pod status into NVRAM
        update_pod_status $ret
        if [ $ret -ne 0 ]; then
             logMesg "=============================================="
             logMesg "=== RP POD FAILED WITH EXIT STATUS' $ret ====="
             logMesg "========= CONTINUING AS BEST-EFFORT! ========="
             logMesg "=============================================="
        else
            # Get the required chassis attribue values for diags.
            DYNATTR_BIN=/usr/lib/siara/bin/dynattr
            echo "product : $(${DYNATTR_BIN} --get chassis/product_code)" >\
                       ${DIAG_CHASSIS_ATTR}
            echo "serial : $(${DYNATTR_BIN}  --get chassis/serial_number)" >>\
                       ${DIAG_CHASSIS_ATTR}
            echo "backplane : $(${DYNATTR_BIN} --get chassis/backplane_id)" >>\
                       ${DIAG_CHASSIS_ATTR}
        fi
    else
        # clear previous pod status into NVRAM
        update_pod_status 0
    fi

    # Initialize the I2C Switching and Locking Infra, ONLY IF POD is happy
    init_i2c_topology

#    # set up routing table 7 used for mgmt interface
     ip rule add pref 32768 table 7 || logMesg "Failed to add table 7 to ipv4 rule"
     ip -6 rule add pref 32760 table 7 || logMesg "Failed to add table 7 to ipv6 rule"

    #publish symlinks for RP thermal sensors on SIM
    if is_nhksim; then
        publish_sim_sensors
    fi

    #set up symlinks for pci devices
    set_pci_symlink

    #set up symlink for i2c
    BOARD_REVISION=$(cat /sys/firmware/rbn/nvenv/chosen/board-revision)
    if [[ $BOARD_REVISION == P02* ]]; then
        symlink_for_i2c
    fi
    rm /dev/cms/cms_ipc
    touch /dev/cms/cms_ipc
}

# Initialize the I2C Switching and Locking Infra.
# This can run ONLY AFTER the switching devices have been blessed by POST/POD
init_i2c_topology() {
    # No-op in SIM
    is_nhksim && return 0

    local bin="/usr/lib/siara/lib64/libnhk_tc_i2cm.so.0.0"
    [ -x "${bin}" ] || return 0

    logMesg "${bin}: Initializing I2C Switching and Locking ..."
    ${bin} --init-sw || logMesg "${bin}: failed - $?"
}

# Set up pci symlink
set_pci_symlink() {
    logMesg "Set up pci symlink"
    local path=/dev/pcie-by-label/
    mkdir -p $path
    local ep
    for ep in /sys/bus/pci/devices/*0.0; do
        vid=$(cat $ep/vendor)
        did=$(cat $ep/device)
        case $vid:$did in
            0x14e4:0x8405) ln -sf $ep $path/ethsw;;
            0x14e4:0x8750) ln -sf $ep $path/fe1600;;
            0x14e4:0x8753) ln -sf $ep $path/fe1600;;
            0x1519:0x0030) ln -sf $ep $path/fpga;;
        esac
    done
}

############################
# Configure M2M interfaces #
############################
m2m_net_config() {

    logMesg "Configuring internal GIGE M2M interface"

    local eth=$($AWK '$1=="ethSw0"{print $NF;}' /proc/net/vlan/config)
    [[ $eth != eth[0-9] ]] && eth="eth1"

    $VCONFIG add ${eth} 108
    $IP link set ${eth}.108 name ethMate0 mtu $GLOBAL_MTU
    $IFCONFIG ethMate0 127.4.$(get_slot_ip).1 netmask 255.255.0.0 up
}

#################################
# Configure Stacking interfaces #
#################################
stacking_net_config() {

    logMesg "Configuring internal GIGE Stacking interface"

    local eth=$($AWK '$1=="ethSw0"{print $NF;}' /proc/net/vlan/config)
    [[ $eth != eth[0-9] ]] && eth="eth1"
    ethmac=$(/usr/lib/siara/bin/dynattr --get chassis/mac_address)
    macdec1=$(expr $((0x${ethmac//:/})) + 2048 - 33 ) # to add 2K and minus LP reserved 32 - 1
    machex1=$(printf "%12x\n" $macdec1 ) # to convert to hex again
    machex2=$(echo $machex1 | sed 's/\(..\)/\1:/g;s/:$//' )


    $VCONFIG add ${eth} 109
    $IP link set ${eth}.109 name ethStack mtu $GLOBAL_MTU
    $IP link set dev ethStack address $machex2
    $IFCONFIG ethStack up
}

#######################
# Configure backplane #
#######################
backplane_net_config() {

    logMesg "Configuring internal GIGE backplane interface..."
    #grep "^igb " /proc/modules >/dev/null 2>&1
    #if [ $? -ne 0 ]; then
        #if [ ! -d /sys/modules/igb ]; then
            #logMesg "Loading igb driver..."
            #/sbin/modprobe igb
        #fi
    #fi

    grep "^bonding " /proc/modules >/dev/null 2>&1
    [[ $? -ne 0 ]] && /sbin/modprobe igb && logMesg "Error: Failed to log igb"

    value=`od -v -t u1 /dev/nvram17 | sed -n '2 p' | awk {'print $17'}`
    ch_id=$((value+1))
    ln -sf $CHASSIS_ID /var/local/chassis_id
    if [ $ch_id != $CHASSIS_ID ]; then
        let input=$((CHASSIS_ID-1))
        hex_a=`printf "%02x" $input`
        printf '\x'${hex_a:0:2}  | dd of=/dev/nvram17 bs=1 seek=31 count=1 conv=notrunc
    fi

    # Hard Coded value for now
    slot=$(get_slot_ip)
    hslot=$(printf "%02x" $slot)

    if is_nhksim ; then
    # Configure VLAN interface - get info from ifLink.data stored by Portal VM
    [ -f /nhksim/var/iflink.data ] && mac=$(grep "IP_PORTAL=" /nhksim/var/iflink.data | cut -d',' -f2) || mac=
    [ ! -z "$mac" ] && eth=$(grep -Hlv "$mac" /sys/class/net/eth[0-9]/address | cut -d/ -f5)

    if [[ $eth == eth[0-9] ]] ; then
        logMesg "$eth is being used for IPC"
    else
        eth="eth0"
        logMesg "Default set to eth0 for IPC"
    fi
    else
    eth="eth1"
        $IP link set dev ${eth} address 02:00:00:01:$hslot:01
    fi

    # Create 2 VLAN devices on ethSw0, Sw0.1
    $VCONFIG add $eth 104 >/dev/null 2>&1
    $VCONFIG add $eth 105 >/dev/null 2>&1
    $IP link set $eth.104 name ethSw0
    $IP link set $eth.105 name ethSw0.1

    $IP link set dev ethSw0 address 02:00:00:01:$hslot:01
    $IP link set dev ethSw0.1 address 02:00:00:01:$hslot:01

    $IFCONFIG $eth mtu $GLOBAL_MTU up >/dev/null 2>&1
    $IFCONFIG ethSw0 127.3.$slot.1 netmask 255.255.0.0 mtu $GLOBAL_MTU up >/dev/null 2>&1
    $IFCONFIG ethSw0.1 127.2.$slot.1 netmask 255.255.0.0 mtu $GLOBAL_MTU up >/dev/null 2>&1

    # echo ethSw0.1 > /sys/class/net/rp0/config/ifname
    echo 67108864 > /proc/sys/net/core/rmem_max
    echo 67108864 > /proc/sys/net/core/wmem_max
    echo 67108864 > /proc/sys/net/core/wmem_default
    echo 67108864 > /proc/sys/net/core/rmem_default

    $IP addr add 127.2.$slot.1/16 brd 127.2.255.255 dev ethSw0.1 scope global
    $IP link set dev $eth up

    # set the TIPC address
    tipc_node=$[slot+1]
    $TIPC_CONFIG -addr=1.1.$tipc_node -be=eth:ethSw0
}

#####################################
# Configure Internal GIGE Interface #
#####################################
internal_gige_net_init() {

    logMesg "Configuring internal GIGE interfaces..."

    # Linecard driver
    # /sbin/modprobe iflc iflc_max_slot=22

    # VLAN driver
    /sbin/modprobe 8021q

    # Bonding driver
    /sbin/modprobe bonding mode=active-backup miimon=1000

    # Loopback Device
    $IFCONFIG lo 127.0.0.1 netmask 255.255.255.0 up

    # Configure GESW interfaces
    backplane_net_config

    # Configure M2M interfaces
    m2m_net_config

    # Configure M2M interfaces
    stacking_net_config

}

###########################################
# print fpga path needed for pci_acccesss #
###########################################
fpga_pci_path_print() {

    FPGA_VENDOR_ID="1519"
    FPGA_DEVICE_ID="0030"
    pci_interface=`lspci  -n  2> /dev/null | grep "$FPGA_VENDOR_ID:$FPGA_DEVICE_ID" | head -1| awk '{print $1}'`
    bus=/sys/bus/pci/devices
    func=resource5
    if [ -z "$pci_interface" ]; then
        logMesg "Error: Could not find tc_fpga, check enumeration."
        return 1
    else
                pci_interface=$bus/0000:${pci_interface}/$func
    fi
    logMesg "pci-id = $pci_interface"
    mkdir -p /dev/tc_fpga/
    if [ $? -ne 0 ]; then
        logMesg "/dev/tc_fpga/ creation failed"
    fi

    ln -s $pci_interface /dev/tc_fpga/pcie_addr
    if [ $? -ne 0 ]; then
        logMesg "Symlink creation failed"
    fi

}

#################################################
# Bind Context Stack on the backplane interface #
# (after attributes have been established)      #
#################################################
backplane_net_config_iflc() {

    logMesg "Configuring Context stack for backplane interface"
    #    $VCONFIG add ethSw0 1

    local iflc_nic=ethSw0.1

    #The '0' is an instance index and has nothing to do with slot id at all
    local iflc_name
        iflc_name="rp0"

    /sbin/modprobe iflc iflc_name="${iflc_name}"

    local slot=$(get_slot_num)
    local host=$((IP_ADDR_OFFSET + slot))
    local pseudo_slot=$(get_slot_ip)

    $IP link set dev ${iflc_nic} down
    echo ${iflc_nic} > /sys/class/net/${iflc_name}/config/ifname
    $IP link set dev ${iflc_nic} up

    $IFCONFIG ${iflc_name} up >/dev/null 2>&1
    $IFCONFIG ${iflc_name} mtu $GLOBAL_MTU >/dev/null 2>&1

}



init_sim_env() {

    logMesg "Booting in NHKSIM environment..."
    hostname ${HOSTNAME}
    /sbin/depmod -a
    /sbin/modprobe loop max_loop=8
    # To make FPGA access via LPC as default bus
    rm -f /etc/siara/fpga_bus_config.cfg
}


fixup_sim_env() {

    # Temporary hack to mount /flash and /var
    mount -t proc proc /proc
    [[ "$?" -ne 0 ]] && logMesg "Failed to mount /proc" && exit 1

    mount -t sysfs sysfs /sys
    [[ "$?" -ne 0 ]] && logMesg "Failed to mount /sys" && exit 1

    mknod /dev/vda3 b $(tr ':' ' ' </sys/class/block/vda3/dev)
    mknod /dev/vda4 b $(tr ':' ' ' </sys/class/block/vda4/dev)

    mount /dev/vda3 /flash
    [ "$?" -ne 0 ] && logMesg "Failed to mount /flash" && exit 1

    mount /dev/vda4 /var
    [ "$?" -ne 0 ] && logMesg "Failed to mount /var" && exit 1

    umount /sys
    umount /proc

    return 0
}


wait_for_link_up() {
    local interface=$1
    local interface_delay=50000
    local interface_max_retry=40
    local retry_count=0

    # Wait upto 2 Seconds
    while [ $retry_count -lt $interface_max_retry ]; do
        usleep $interface_delay
        local link_status=`ip link show dev $interface | grep LOWER_UP >/dev/null 2>&1`
        [[ $? -eq 0 ]] && return 0
        retry_count=$((retry_count+1))
    done
    local timeout=$((interface_delay * $interface_max_retry))
    logMesg "Error: Link status is still down for interface $1, timeout =  $timeout micro-second"
    return 1
}


rename_interface() {
    $IP link set $1 name $2
    if [ $? -ne 0 ]; then
        logMesg "Error: Could not rename interface $1 to $2"
        return 1
    fi
    return 0
}


create_vfs_on_PF() {
    local pf_index=$1
    local pf_name=$2
    local num_vfs=$3

    logMesg "pf_index = $pf_index pf_name = $pf_name"
    local name=$(ls /sys/module/ixgbe/drivers/pci\:ixgbe/0000:*.${pf_index}/net 2>/dev/null)
    if [ $? -ne 0 ]; then
        logMesg "Error: Invalid pf_index $pf_index for $pf_name"
        return 1
    fi

    $IP link set $name down
    rename_interface $name $pf_name
    if [ $? -ne 0 ]; then
        return 1
    fi

    $IP link set $pf_name up

    ## We must have this delay into wait_for_link_up fuction
    ## to allow PF Mailbox to be ready before
    ## vfs are created and access mailbox to communicate with PF
    wait_for_link_up $pf_name
    if [ $? -ne 0 ]; then
        return 1
    fi

    ## we must load ixgbevf only after PF interface is up
    ## Why we still need to find root cause TODO???
    lsmod | grep ixgbevf > /dev/null
    if [ $? -ne 0 ] ; then
        /sbin/modprobe ixgbevf
        if [ $? -ne 0 ] ; then
            logMesg "Failed to load ixgbevf driver"
            return 1
        fi
    fi

    ## We must have this delay into wait_for_link_up fuction
    ## to allow PF Mailbox to be ready before
    ## vfs are created and access mailbox to communicate with PF
    ## Without this delay we see a very intermittent issue where vf0
    ## creation fails as VF driver believes PF is in reset. vf1/2
    ## creation is however successful. With the additional delay,
    ## this issue is not reproducible.
    wait_for_link_up $pf_name
    if [ $? -ne 0 ]; then
        return 1
    fi

    echo "$num_vfs" > /sys/class/net/$pf_name/device/sriov_numvfs
    if [ $? -ne 0 ]; then
        logMesg "Error: Could not create $num_vfs vfs"
        return 1;
    fi
    return 0;
}


config_vfs() {

    local pf_name=$1
    local vf_index=$2
    local vlan_tag=$3
    local qos=$4
    local vf_new_name=$5

    if [ $# -ne 5 ]; then
        logMesg  "Error: not enough arguments to config_vfs"
        logMesg  " config_vfs <pf_name> <vf_index> <vlan_tag> <qos> <vf_new_name>"
        return 1
    fi

    local vf_name=`ls /sys/class/net/$pf_name/device/virtfn${vf_index}/net`
        echo "PFNAME : $pf_name vf_name = $vf_name"

    if [ $? -eq 0 ]; then
        if [ -z $vf_name ]; then
            logMesg "Error: Failed to get VF name ."
        fi

        $IP link set $pf_name vf $vf_index vlan $vlan_tag qos $qos spoofchk on
        if [ $? -ne 0 ]; then
            logMesg  "Error: Could not set $pf_name vf $vf_index vlan \
            $vlan_tag qos $qos spoofchk on"
            return 1
        fi

        rename_interface $vf_name  $vf_new_name
        if [ $? -ne 0 ]; then
            return 1
        fi
    fi

    return 0
}

front_gige_net_init() {

        #bring up eth0 physical interface of RP
        ifconfig eth0 up

    # ethernet index
    # creatte_vfs_on_PF pf_index pf_name num_vfs
    create_vfs_on_PF 0 ethFP 3
    if [ $? -ne 0 ]; then
        logMesg "Error: Failed to create VFS(ethFP)."
            return 1
    fi

    # creatte_vfs pf_name vf_index vlan_tag qos vf_new_name
    # ethSDN, and ethCTL remains down. CLI will bring eth0 up later.
    config_vfs ethFP 1 102 6 ethSDN
    if [ $? -ne 0 ]; then
        logMesg "Error: Failed to create VFS(ethSDN)."
        return 1;
    fi

    config_vfs ethFP 2 103 6 ethCTL
    if [ $? -ne 0 ]; then
        logMesg "Error: Failed to create VFS(ethCTL)."
        return 1;
    fi

    config_vfs ethFP 0 101 7 eth0
    if [ $? -ne 0 ]; then
        logMesg "Error: Failed to configure VFS(ethFP)"
        return 1;
    fi

    # assign MAC address to vf 0 (eth0)
    local vf0_mac=`cat /sys/firmware/rbn/nvenv/mac-address`
    logMesg "vf 0 mac $vf0_mac"
    $IP link set dev ethFP vf 0 mac $vf0_mac
    if [ $? -ne 0 ]; then
        logMesg "can't set MAC:$vf0_mac to ethFP vf 0"
        return 1;
    fi

}

NHK_no_stacking() {
    if [ $STACKING_ENABLE -eq 0 ]; then
        echo "No stacking configuration... "
        mkdir -p /dev/cms/bcm/config
        ln -sf disable /dev/cms/bcm/config/stacking
   fi
}

check_mac_address () {
    is_nhksim && return 0
    EEPROM_MAC=`dd if=/dev/mem bs=1 skip=$((0xad056)) count=6 | xxd -ps -u`
    echo EEPROM_MAC: $EEPROM_MAC
    FWENV_MAC=`cat /sys/firmware/rbn/nvenv/mac-address | sed 's/://g'`
    echo FWENV_MAC: $FWENV_MAC

    if [ "$EEPROM_MAC" = "$FWENV_MAC" ]; then
        echo "MAC address in NVRAM and FWENV matches"
    else
        echo "MAC address in EEPROM and NVRAM FWENV doesn't matches"
        echo "Rebooting system"
        /usr/lib/siara/sbin/rbn_reboot
    fi;
}

get_chassis_id() {
  # Create needed directories for swarbiter in presistent partition and create symbolic link.
  [ -d /var/systems/confd ] || mkdir -p /var/systems/confd

  # Create a folder used by nhk, not affected by software upgrade
  [ -d /var/systems/nhk ] || mkdir -p -m 0755 /var/systems/nhk

  # copy old file if it exist
  [[ -f /var/systems/confd/swarbiter.cfg ]] &&
      mv -f /var/systems/confd/swarbiter.cfg /var/systems/nhk/

  if [ -f /var/systems/nhk/swarbiter.cfg ]; then
    CHASSIS_ID=$(grep Chassis-ID /var/systems/nhk/swarbiter.cfg | $AWK -F":" '{print $2}')
    if [ -z "$CHASSIS_ID" ]; then
       echo "CANNOT DERIVE CHASSIS ID, SETTING MY CHASSIS = 1";
       CHASSIS_ID=1;
    fi;
    STACK_STATUS=$(grep Stack-Enable /var/systems/nhk/swarbiter.cfg | $AWK -F":" '{print $2}')
    if [ -z "$STACK_STATUS" ]; then
        echo "CANNOT DERIVE STACKING STATUS, SETTING STACKING_ENABLE = 0";
        STACKING_ENABLE=0;
    elif [ "$STACK_STATUS" == "False" ]; then
        STACKING_ENABLE=0;
    else
        STACKING_ENABLE=1;
    fi
  fi;

  echo "MY CHASSIS WILL BE $CHASSIS_ID, STACKING STATUS WILL BE $STACKING_ENABLE";
}

######################
## Main Begins Here ##
######################
main() {
    logMesg "Started Linux initialization script on $(date -u)"

    if is_nhksim; then
        # use /var/lock to figure out whether the new VM images are being
        # used or not. Specifically, /var/lock is being created in vda4
        # partition by new VM qcow images, instead of in vda1.

        # if we see /var/lock now, it must mean it exists in vda1 ...
        if [ -d /var/lock ] ; then
            logMesg "Old versions of VM images are being used ..."
        else
            logMesg "New versions of VM images are being used ..."
            fixup_sim_env
        fi

        GLOBAL_MTU=1500

        init_sim_env
    fi

    logMesg "Global MTU set to $GLOBAL_MTU"

    # Re-initialize the log file
    rotateLogFiles boot.log

    # Setup file-system related stuff
    setup_fs_linux

    # Check if MAC addr is valid or not
    check_mac_address

    setup_fs_ipos

    # Extract chassis ID
    get_chassis_id

    # Configure linux
    configure_linux

    # Initialize IPOS related stuff
    rbos_init

    #internal_vfab_net_init

    # Initialize dynattr infrastructure and publish vital attributes.
    platform_init

    # All hardware specific configuration should happen after POD.
    # Configure gige net interface
    internal_gige_net_init

    #Create pci_id symlink for fpga driver
    fpga_pci_path_print

    # PF ethFP is front gige
    front_gige_net_init

    # Construct the stack for supporting the RBN context stack
    backplane_net_config_iflc

    #PLATFORM_NHK_STACKING
    NHK_no_stacking

    # Update Mailbox 1 for healthy status
    /usr/lib/siara/bin/mailbox_write $MAILBOX_1 $CARD_UP
}

main

# Setup Iptable rules
/etc/iptables-rules.sh

# Apply file system hardening (FSH) facl policy
/usr/lib/siara/bin/facl_policy_apply.sh

if [ "${BOOT_MODE}" != "diag" ]; then
   # TODO, need to move this to some process so
   # it can be monitored, Monitor chassis ID
   /usr/lib/siara/bin/chassis_id_monitor.sh &
fi

# Collect NVRAM logs into /var/log folder
echo ">>>>>>>>>>>$(date -u)<<<<<<<<<<<<<<<" >> /var/log/rp_console.log
/bin/dd if=/dev/nvram15  >> /var/log/rp_console.log

# Collect RP Reset cause values in /var/log folder
/usr/lib/siara/bin/reset_cause.py /dev/nvram16 >> /var/log/rp_console.log
echo ">>>>>>>>>>>$(date -u)<<<<<<<<<<<<<<<" >> /var/log/rp_console.log
rm /usr/lib/siara/bin/cms_ipc
touch /usr/lib/siara/bin/cms_ipc

exit 0

